<?php
/**
 * Logs
 *
 * @package     AutomatorWP\Restrict Content\Logs
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * Adds user to access level
 *
 * @since 1.0.0
 *
 * @return bool true on delete, false on failure
 */
function automatorwp_restrict_content_add_user_to_access_level( $user_id, $access_level_id ) {

    global $wpdb;

    $user_id = absint( $user_id );
    $access_level_id = absint( $access_level_id );

    if( automatorwp_restrict_content_is_user_on_access_level( $user_id, $access_level_id ) ) {
        return false; // User already on level
    }

    // Setup table
    $ct_table = ct_setup_table( 'automatorwp_restrict_content_access_level_users' );

    ct_insert_object( array(
        'user_id' => $user_id,
        'access_level_id' => $access_level_id,
    ) );

    ct_reset_setup_table();

    return true;

}

/**
 * Removes user from level
 *
 * @since 1.0.0
 *
 * @return bool true on delete, false on failure
 */
function automatorwp_restrict_content_remove_user_from_access_level( $user_id, $access_level_id ) {

    global $wpdb;

    $user_id = absint( $user_id );
    $access_level_id = absint( $access_level_id );

    $access_level = automatorwp_restrict_content_get_access_level_entry( $user_id, $access_level_id );

    if( ! $access_level ) {
        return false; // User is not in level so can not remove it
    }

    // Setup table
    $ct_table = ct_setup_table( 'automatorwp_restrict_content_access_level_users' );
    
    ct_delete_object( absint( $access_level->id ) );
    $deleted = $wpdb->delete( $ct_table->db->table_name, array( 'id' => $access_level->id ), array( '%d' ) );

    ct_reset_setup_table();

    return true;

}

/**
 * Checks if user is on the given level
 *
 * @since 1.0.0
 *
 * @return bool
 */
function automatorwp_restrict_content_is_user_on_access_level( $user_id, $access_level_id ) {

    global $wpdb;

    $user_id = absint( $user_id );
    $access_level_id = absint( $access_level_id );

    // Setup table
    $ct_table = ct_setup_table( 'automatorwp_restrict_content_access_level_users' );

    $count = absint( $wpdb->get_var(
        "SELECT COUNT(*) 
                FROM {$ct_table->db->table_name} AS ulu 
                WHERE ulu.user_id = {$user_id}
                AND ulu.access_level_id = {$access_level_id}"
    ) );

    ct_reset_setup_table();

    return (bool) ( $count >= 1 );

}

/**
 * Checks if user is on the given level
 *
 * @since 1.0.0
 *
 * @return stdClass The user level entry object (id, level_id, user_id)
 */
function automatorwp_restrict_content_get_access_level_entry( $user_id, $access_level_id ) {

    global $wpdb;

    $user_id = absint( $user_id );
    $access_level_id = absint( $access_level_id );

    // Setup table
    $ct_table = ct_setup_table( 'automatorwp_restrict_content_access_level_users' );

    $access_level = $wpdb->get_row(
        "SELECT *
                FROM {$ct_table->db->table_name} AS ulu 
                WHERE ulu.user_id = {$user_id}
                AND ulu.access_level_id = {$access_level_id}"
    );

    ct_reset_setup_table();

    return $access_level;

}

/**
 * Checks if user is on the given level
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_restrict_content_get_user_access_levels( $user_id ) {

    global $wpdb;

    $user_id = absint( $user_id );

    // Setup table
    $ct_table = ct_setup_table( 'automatorwp_restrict_content_access_level_users' );
    $access_level_user_table = $ct_table->db->table_name;
    ct_reset_setup_table();

    $ct_table = ct_setup_table( 'automatorwp_restrict_content_access_levels' );

    $access_level = $wpdb->get_results(
        "SELECT ul.*
                FROM {$ct_table->db->table_name} AS ul
                LEFT JOIN {$access_level_user_table} AS ulu ON ulu.access_level_id = ul.id
                WHERE ulu.user_id = {$user_id}"
    );

    ct_reset_setup_table();

    return $access_level;

}

/**
 * Get all custom user fields
 *
 * @since 1.0.0
 *
 * @return array
 */
function automatorwp_restrict_content_get_all_access_levels() {

    global $wpdb, $automatorwp_restrict_content_cache;

    if( ! $automatorwp_restrict_content_cache ) {

        // Setup table
        $ct_table = ct_setup_table( 'automatorwp_restrict_content_access_levels' );

        // Get all fields
        $automatorwp_restrict_content_cache = $wpdb->get_results( "SELECT * FROM {$ct_table->db->table_name} AS ul" );

        ct_reset_setup_table();

    }

    return $automatorwp_restrict_content_cache;

}

/**
 * Get the count of all custom user fields
 *
 * @since 1.0.0
 *
 * @return int
 */
function automatorwp_restrict_content_get_access_level_user_count( $access_level_id ) {

    global $wpdb;

    $access_level_id = absint( $access_level_id );

    // Setup table
    $ct_table = ct_setup_table( 'automatorwp_restrict_content_access_level_users' );

    $count = absint( $wpdb->get_var( "SELECT COUNT(*) FROM {$ct_table->db->table_name} AS ulu WHERE ulu.access_level_id = {$access_level_id}" ) );

    ct_reset_setup_table();

    return $count;

}

/**
 * Find custom user field that matches with a meta key
 *
 * @since 1.0.0
 *
 * @param string $name The level name to search
 *
 * @return array
 */
function automatorwp_restrict_content_get_access_level_by_name( $name ) {

    global $wpdb;

    // Setup table
    $ct_table = ct_setup_table( 'automatorwp_restrict_content_access_levels' );

    // Search custom field by meta key
    $custom_field = $wpdb->get_row( $wpdb->prepare(
        "SELECT *
        FROM {$ct_table->db->table_name} AS ul
        WHERE ul.name = %s",
        $name,
    ) );

    ct_reset_setup_table();

    return $custom_field;

}

/**
 * Find custom user field that matches with a meta key
 *
 * @since 1.0.0
 *
 * @param string $slug The level slug to search
 *
 * @return array
 */
function automatorwp_restrict_content_get_access_level_by_slug( $slug ) {

    global $wpdb;

    // Setup table
    $ct_table = ct_setup_table( 'automatorwp_restrict_content_access_levels' );

    // Search custom field by meta key
    $custom_field = $wpdb->get_row( $wpdb->prepare(
        "SELECT *
        FROM {$ct_table->db->table_name} AS ul
        WHERE ul.slug = %s",
        $slug,
    ) );

    ct_reset_setup_table();

    return $custom_field;

}

/**
 * Find custom user field that matches with a level ID
 *
 * @since 1.0.0
 *
 * @param int $access_level_id The level ID to search
 *
 * @return array
 */
function automatorwp_restrict_content_get_access_level_by_id( $access_level_id ) {

    global $wpdb;

    // Setup table
    $ct_table = ct_setup_table( 'automatorwp_restrict_content_access_levels' );

    // Search level by level ID
    $custom_field = $wpdb->get_row( $wpdb->prepare(
        "SELECT *
        FROM {$ct_table->db->table_name} AS ul
        WHERE ul.id = %s",
        $access_level_id,
    ) );

    ct_reset_setup_table();

    return $custom_field;

}
